<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Keygen;
use App\Brand;
use App\Category;
use App\Unit;
use App\Tax;
use App\Warehouse;
use App\Supplier;
use App\Product;
use App\Color;
use App\Size;
use App\ProductBatch;
use App\Product_Warehouse;
use App\Product_Supplier;
use App\SalesPayment;
use Auth;
use DNS1D;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use Illuminate\Validation\Rule;
use DB;
use App\Variant;
use App\BarcodeSetting;
use App\ProductVariant;
use Picqer\Barcode\BarcodeGeneratorPNG;
use Illuminate\Support\Str;

class ProductController extends Controller
{
    public function index()
    {
        $role = Role::find(Auth::user()->role_id);
        if($role->hasPermissionTo('products-index')){            
            $permissions = Role::findByName($role->name)->permissions;
            foreach ($permissions as $permission)
                $all_permission[] = $permission->name;
            if(empty($all_permission))
                $all_permission[] = 'dummy text';
            return view('product.index', compact('all_permission'));
        }
        else
            return redirect()->back()->with('not_permitted', 'Sorry! You are not allowed to access this module');
    }
      public function productData(Request $request)
    {
        $columns = array( 
            2 => 'name', 
            3 => 'code',
            4 => 'brand_id',
            5 => 'category_id',
            6 => 'qty',
            7 => 'unit_id',
            8 => 'price',
            9 => 'cost',
            10 => 'stock_worth',
            11 => 'created_at'
        );
        
        $totalData = Product::where('is_active', true)->count();
        $totalFiltered = $totalData; 

        if($request->input('length') != -1)
            $limit = $request->input('length');
        else
            $limit = $totalData;
        $start = $request->input('start');
        $order = 'products.'.$columns[$request->input('order.0.column')];
        $dir = $request->input('order.0.dir');
        if(empty($request->input('search.value'))){
            $products = Product::with('category', 'brand', 'unit')->offset($start)
                        ->where('is_active', true)
                        ->limit($limit)
                        ->orderBy($order,$dir)
                        ->get();
        }
        else
        {
            $search = $request->input('search.value'); 
            $products =  Product::select('products.*')
                        ->with('category', 'brand', 'unit')
                        ->join('categories', 'products.category_id', '=', 'categories.id')
                        ->leftjoin('brands', 'products.brand_id', '=', 'brands.id')
                        ->where([
                            ['products.name', 'LIKE', "%{$search}%"],
                            ['products.is_active', true]
                        ])
                        ->orWhere([
                            ['products.code', 'LIKE', "%{$search}%"],
                            ['products.is_active', true]
                        ])
                        ->orWhere([
                            ['categories.name', 'LIKE', "%{$search}%"],
                            ['categories.is_active', true],
                            ['products.is_active', true]
                        ])
                        ->orWhere([
                            ['brands.title', 'LIKE', "%{$search}%"],
                            ['brands.is_active', true],
                            ['products.is_active', true]
                        ])
                        ->offset($start)
                        ->limit($limit)
                        ->orderBy($order,$dir)->get();

            $totalFiltered = Product::
                            join('categories', 'products.category_id', '=', 'categories.id')
                            ->leftjoin('brands', 'products.brand_id', '=', 'brands.id')
                            ->where([
                                ['products.name','LIKE',"%{$search}%"],
                                ['products.is_active', true]
                            ])
                            ->orWhere([
                                ['products.code', 'LIKE', "%{$search}%"],
                                ['products.is_active', true]
                            ])
                            ->orWhere([
                                ['categories.name', 'LIKE', "%{$search}%"],
                                ['categories.is_active', true],
                                ['products.is_active', true]
                            ])
                            ->orWhere([
                                ['brands.title', 'LIKE', "%{$search}%"],
                                ['brands.is_active', true],
                                ['products.is_active', true]
                            ])
                            ->count();
        }
        $data = array();
        if(!empty($products))
        {
            foreach ($products as $key=>$product)
            {
                
                $nestedData['id'] = $product->id;
                $nestedData['key'] = $key;
                $product_image = explode(",", $product->image);
                $product_image = htmlspecialchars($product_image[0]);
                $nestedData['image'] = '<img src="'.url('public/images/product', $product_image).'" height="80" width="80">';
                $nestedData['name'] = $product->name;
                $nestedData['code'] = $product->code;
                if($product->brand_id)
                    $nestedData['brand'] = $product->brand->title;
                else
                $nestedData['brand'] = "N/A";
                $nestedData['category'] = $product->category->name;
                if ($product->is_imei !== null) {
                    $unit = Unit::where('base_unit', $product->unit_id)->first();
                    $operater_name  = $unit->unit_name;
                    $oparater       = $unit->operator;
                    $oparater_value = $unit->operation_value;
                
                    if ($oparater == '*') {
                        $qty = $product->qty * $oparater_value;
                    } elseif ($oparater == '/') {
                        $qty = $product->qty / $oparater_value;
                    } else {
                        $qty = $product->qty;
                    }
                
                    $nestedData['qty'] = number_format($qty, 2) . ' ' . $operater_name;
                
                } else {
                    $unit = Unit::where('id', $product->unit_id)->first();
                    $nestedData['qty'] = number_format($product->qty, 2) . ' ' . $unit->unit_code;
                }


                // $nestedData['qty'] = $product->is_imei !== null
                // ? ($product->unit->operator == '*'
                //     ? $product->qty * $product->unit->operation_value
                //     : $product->qty / $product->unit->operation_value)
                // : $product->qty;

                if($product->unit_id)
                    $nestedData['unit'] = $product->unit->unit_name;
                else
                    $nestedData['unit'] = 'N/A';
                
                $nestedData['price'] = $product->price;
                $nestedData['cost'] = $product->cost;
                $nestedData['created_at'] = $product->created_at ? $product->created_at->format('Y-m-d H:i:s') : 'N/A';

                if(config('currency_position') == 'prefix')
                    $nestedData['stock_worth'] = config('currency').' '.($product->qty * $product->price).' / '.config('currency').' '.($product->qty * $product->cost);
                else
                    $nestedData['stock_worth'] = ($product->qty * $product->price).' '.config('currency').' / '.($product->qty * $product->cost).' '.config('currency');
                //$nestedData['stock_worth'] = ($product->qty * $product->price).'/'.($product->qty * $product->cost);

                $nestedData['options'] = '<div class="btn-group">
                            <button type="button" class="btn btn-default dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">'.trans("file.action").'
                              <span class="caret"></span>
                              <span class="sr-only">Toggle Dropdown</span>
                            </button>
                            <ul class="dropdown-menu edit-options dropdown-menu-right dropdown-default" user="menu">
                            <li>
                                <button="type" class="btn btn-link view"><i class="fa fa-eye"></i> '.trans('file.View').'</button>
                            </li>';
                if(in_array("products-edit", $request['all_permission']))
                    $nestedData['options'] .= '<li>
                            <a href="'.route('products.edit', $product->id).'" class="btn btn-link"><i class="fa fa-edit"></i> '.trans('file.edit').'</a>
                        </li>';
                if(in_array("products-delete", $request['all_permission']))
                    $nestedData['options'] .= \Form::open(["route" => ["products.destroy", $product->id], "method" => "DELETE"] ).'
                            <li>
                              <button type="submit" class="btn btn-link" onclick="return confirmDelete()"><i class="fa fa-trash"></i> '.trans("file.delete").'</button> 
                            </li>'.\Form::close().'
                        </ul>
                    </div>';
                // data for product details by one click
                if($product->tax_id)
                    $tax = Tax::find($product->tax_id)->name;
                else
                    $tax = "N/A";

                if($product->tax_method == 1)
                    $tax_method = trans('file.Exclusive');
                else
                    $tax_method = trans('file.Inclusive');

                $nestedData['product'] = array( '[ "'.$product->type.'"', ' "'.$product->name.'"', ' "'.$product->code.'"', ' "'.$nestedData['brand'].'"', ' "'.$nestedData['category'].'"', ' "'.$nestedData['unit'].'"', ' "'.$product->cost.'"', ' "'.$product->price.'"', ' "'.$tax.'"', ' "'.$tax_method.'"', ' "'.$product->alert_quantity.'"', ' "'.preg_replace('/\s+/S', " ", $product->product_details).'"', ' "'.$product->id.'"', ' "'.$product->product_list.'"', ' "'.$product->variant_list.'"', ' "'.$product->qty_list.'"', ' "'.$product->price_list.'"', ' "'.$product->qty.'"',' "' . $product->image . '"',
                ' "' . $product->created_at->format('Y-m-d H:i:s') .'" ]'
                );
                //$nestedData['imagedata'] = DNS1D::getBarcodePNG($product->code, $product->barcode_symbology);
                $data[] = $nestedData;
            }
        }
        $json_data = array(
            "draw"            => intval($request->input('draw')),  
            "recordsTotal"    => intval($totalData),  
            "recordsFiltered" => intval($totalFiltered), 
            "data"            => $data   
        );
            
        echo json_encode($json_data);
    }
    
    
//     public function productData(Request $request)
// {
//     $columns = array( 
//         2 => 'name', 
//         3 => 'code',
//         4 => 'brand_id',
//         5 => 'category_id',
//         6 => 'qty',
//         7 => 'unit_id',
//         8 => 'price',
//         9 => 'cost',
//         10 => 'stock_worth',
//         11 => 'created_at'
//     );

//     $start = $request->input('start');
//     $limit = $request->input('length') != -1 ? $request->input('length') : Product::where('is_active', true)->count();
//     $order = 'products.' . $columns[$request->input('order.0.column')];
//     $dir = $request->input('order.0.dir');

//     $baseFilter = function ($query) {
//         $query->where('code', 'NOT LIKE', '%-%')
//               ->orWhere('code', 'LIKE', '%-001-%');
//     };

//     if (empty($request->input('search.value'))) {
//         $products = Product::with('category', 'brand', 'unit')
//             ->where('is_active', true)
//             ->where($baseFilter)
//             ->offset($start)
//             ->limit($limit)
//             ->orderBy($order, $dir)
//             ->get();

//         $totalData = Product::where('is_active', true)->where($baseFilter)->count();
//         $totalFiltered = $totalData;
//     } else {
//         $search = $request->input('search.value');

//         $products = Product::select('products.*')
//             ->with('category', 'brand', 'unit')
//             ->join('categories', 'products.category_id', '=', 'categories.id')
//             ->leftJoin('brands', 'products.brand_id', '=', 'brands.id')
//             ->where('products.is_active', true)
//             ->where($baseFilter)
//             ->where(function ($query) use ($search) {
//                 $query->where('products.name', 'LIKE', "%{$search}%")
//                       ->orWhere('products.code', 'LIKE', "%{$search}%")
//                       ->orWhere('categories.name', 'LIKE', "%{$search}%")
//                       ->orWhere('brands.title', 'LIKE', "%{$search}%");
//             })
//             ->offset($start)
//             ->limit($limit)
//             ->orderBy($order, $dir)
//             ->get();

//         $totalFiltered = Product::join('categories', 'products.category_id', '=', 'categories.id')
//             ->leftJoin('brands', 'products.brand_id', '=', 'brands.id')
//             ->where('products.is_active', true)
//             ->where($baseFilter)
//             ->where(function ($query) use ($search) {
//                 $query->where('products.name', 'LIKE', "%{$search}%")
//                       ->orWhere('products.code', 'LIKE', "%{$search}%")
//                       ->orWhere('categories.name', 'LIKE', "%{$search}%")
//                       ->orWhere('brands.title', 'LIKE', "%{$search}%");
//             })
//             ->count();

//         $totalData = Product::where('is_active', true)->count(); // still show total
//     }

//     $data = array();
//     if (!empty($products)) {
//         foreach ($products as $key => $product) {
//             $nestedData['id'] = $product->id;
//             $nestedData['key'] = $key;
//             $product_image = explode(",", $product->image);
//             $product_image = htmlspecialchars($product_image[0]);
//             $nestedData['image'] = '<img src="' . url('public/images/product', $product_image) . '" height="80" width="80">';
//             $nestedData['name'] = str_replace('-1', '', $product->name);
//             $nestedData['code'] = explode('-', $product->code)[0];
//             $nestedData['brand'] = $product->brand_id ? $product->brand->title : "N/A";
//             $nestedData['category'] = $product->category->name;
//             $nestedData['qty'] = $product->qty;
//             $nestedData['unit'] = $product->unit_id ? $product->unit->unit_name : 'N/A';
//             $nestedData['price'] = $product->price;
//             $nestedData['cost'] = $product->cost;
//              $nestedData['created_at'] = $product->created_at ? $product->created_at->format('Y-m-d H:i:s') : 'N/A';
//             if (config('currency_position') == 'prefix')
//                 $nestedData['stock_worth'] = config('currency') . ' ' . ($product->qty * $product->price) . ' / ' . config('currency') . ' ' . ($product->qty * $product->cost);
//             else
//                 $nestedData['stock_worth'] = ($product->qty * $product->price) . ' ' . config('currency') . ' / ' . ($product->qty * $product->cost) . ' ' . config('currency');

//             $nestedData['options'] = '<div class="btn-group">
//                 <button type="button" class="btn btn-default dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">' . trans("file.action") . '
//                     <span class="caret"></span>
//                     <span class="sr-only">Toggle Dropdown</span>
//                 </button>
//                 <ul class="dropdown-menu edit-options dropdown-menu-right dropdown-default" user="menu">
//                     <li>
//                         <button type="button" class="btn btn-link view"><i class="fa fa-eye"></i> ' . trans('file.View') . '</button>
//                     </li>';
//             if (in_array("products-edit", $request['all_permission']))
//                 $nestedData['options'] .= '<li>
//                     <a href="' . route('products.edit', $product->id) . '" class="btn btn-link"><i class="fa fa-edit"></i> ' . trans('file.edit') . '</a>
//                 </li>';
//             if (in_array("products-delete", $request['all_permission']))
//                 $nestedData['options'] .= \Form::open(["route" => ["products.destroy", $product->id], "method" => "DELETE"]) . '
//                     <li>
//                         <button type="submit" class="btn btn-link" onclick="return confirmDelete()"><i class="fa fa-trash"></i> ' . trans("file.delete") . '</button> 
//                     </li>' . \Form::close() . '
//                 </ul>
//             </div>';

//             $tax = $product->tax_id ? Tax::find($product->tax_id)->name : "N/A";
//             $tax_method = $product->tax_method == 1 ? trans('file.Exclusive') : trans('file.Inclusive');

//             $nestedData['product'] = array(
//                 '[ "' . $product->type . '"',
//                 ' "' . str_replace('-1', '', $product->name) . '"',
//                 ' "' . explode('-', $product->code)[0] . '"',
//                 ' "' . $nestedData['brand'] . '"',
//                 ' "' . $nestedData['category'] . '"',
//                 ' "' . $nestedData['unit'] . '"',
//                 ' "' . $product->cost . '"',
//                 ' "' . $product->price . '"',
//                 ' "' . $tax . '"',
//                 ' "' . $tax_method . '"',
//                 ' "' . $product->alert_quantity . '"',
//                 ' "' . preg_replace('/\s+/S', " ", $product->product_details) . '"',
//                 ' "' . $product->id . '"',
//                 ' "' . $product->product_list . '"',
//                 ' "' . $product->variant_list . '"',
//                 ' "' . $product->qty_list . '"',
//                 ' "' . $product->price_list . '"',
//                 ' "' . $product->qty . '"',
//                 ' "' . $product->image . '"',
//                 ' "' . $product->created_at->format('Y-m-d H:i:s') .'" ]'
//             );

//             $data[] = $nestedData;
//         }
//     }

//     return response()->json([
//         "draw" => intval($request->input('draw')),
//         "recordsTotal" => intval($totalData),
//         "recordsFiltered" => intval($totalFiltered),
//         "data" => $data
//     ]);
// }


    public function create()
    {
        $role = Role::firstOrCreate(['id' => Auth::user()->role_id]);
        if ($role->hasPermissionTo('products-add')){
            $lims_product_list_without_variant = $this->productWithoutVariant();
            $lims_product_list_with_variant = $this->productWithVariant();
            $lims_brand_list = Brand::where('is_active', true)->get();
            $lims_category_list = Category::where('is_active', true)->get();
            $lims_unit_list = Unit::where('is_active', true)->get();
            $lims_tax_list = Tax::where('is_active', true)->get();
            $lims_warehouse_list = Warehouse::where('is_active', true)->get();
            $colors = Color::where('is_active', true)->get();
            $sizes = Size::where('is_active',true)->get();
            return view('product.create',compact('lims_product_list_without_variant', 'lims_product_list_with_variant', 'lims_brand_list', 'lims_category_list', 'lims_unit_list', 'lims_tax_list', 'lims_warehouse_list','colors','sizes'));
        }
        else
            return redirect()->back()->with('not_permitted', 'Sorry! You are not allowed to access this module');
    }
     public function store(Request $request)
    {
        $this->validate($request, [
            'code' => [
                'max:255',
                    Rule::unique('products')->where(function ($query) {
                    return $query->where('is_active', 1);
                }),
            ],
            'name' => [
                'max:255',
                    Rule::unique('products')->where(function ($query) {
                    return $query->where('is_active', 1);
                }),
            ]
        ]);
        $data = $request->except('image', 'file');
        $data['name'] = htmlspecialchars(trim($data['name']));
        if($data['type'] == 'combo'){
            $data['product_list'] = implode(",", $data['product_id']);
            $data['variant_list'] = implode(",", $data['variant_id']);
            $data['qty_list'] = implode(",", $data['product_qty']);
            $data['price_list'] = implode(",", $data['unit_price']);
            $data['cost'] = $data['unit_id'] = $data['purchase_unit_id'] = $data['sale_unit_id'] = 0;
        }
        elseif($data['type'] == 'digital' || $data['type'] == 'service')
            $data['cost'] = $data['unit_id'] = $data['purchase_unit_id'] = $data['sale_unit_id'] = 0;

        $data['product_details'] = str_replace('"', '@', $data['product_details']);

        if($data['starting_date'])
            $data['starting_date'] = date('Y-m-d', strtotime($data['starting_date']));
        if($data['last_date'])
            $data['last_date'] = date('Y-m-d', strtotime($data['last_date']));
        $data['is_active'] = true;
        $images = $request->image;
        $image_names = [];
        if($images) {            
            foreach ($images as $key => $image) {
                $imageName = $image->getClientOriginalName();
                $image->move('public/images/product', $imageName);
                $image_names[] = $imageName;
            }
            $data['image'] = implode(",", $image_names);
        }
        else {
            $data['image'] = 'zummXD2dvAtI.png';
        }
        $file = $request->file;
        if ($file) {
            $ext = pathinfo($file->getClientOriginalName(), PATHINFO_EXTENSION);
            $fileName = strtotime(date('Y-m-d H:i:s'));
            $fileName = $fileName . '.' . $ext;
            $file->move('public/product/files', $fileName);
            $data['file'] = $fileName;
        }
        
         $variantImages = $request->file('variants_image'); // array of files
        $variantImagePaths = [];
    
        if (!empty($variantImages)) {
            foreach ($variantImages as $variantImage) {
                if ($variantImage) {
                    $imageName = time() . '_' . $variantImage->getClientOriginalName();
                    $variantImage->move(public_path('images/product'), $imageName);
                    $variantImagePaths[] = $imageName;
                } else {
                    $variantImagePaths[] = null;
                }
            }
        }


  
        $products = [];
      if (!empty($data['variants_name']) && is_array($data['variants_name'])) {
        $products = [];
            foreach ($data['variants_name'] as $key => $variantName) {
                
                $variantData = [
                    'type' => $data['type'],
                    'name' => $variantName,
                    'color' => $data['variants_color'][$key] ?? null,
                    'size' => $data['variants_size'][$key] ?? null,
                    'code' => $data['variants_item_code'][$key] ?? null,
                    'barcode_symbology' => $data['barcode_symbology'],
                    'brand_id' => $data['brand_id'] ?? null,
                    'category_id' => $data['category_id'] ?? null,
                    'unit_id' => $data['unit_id'],
                    'sale_unit_id' => $data['sale_unit_id'],
                    'purchase_unit_id' => $data['purchase_unit_id'],
                    'cost' => $data['variant_purchase_price'][$key] ?? 0,
                    'price' => $data['variants_sale_price'][$key] ?? 0,
                    'qty' => $data['variants_qty'][$key] ?? 0,
                    'alert_quantity' => $data['alert_quantity'],
                    'tax_id' => $data['tax_id'] ?? null,
                    'tax_method' => $data['tax_method'] ?? null,
                    'featured' => $data['featured'] ?? 0,
                    'is_variant' => null,
                    'image' => $variantImagePaths[$key] ?? 'zummXD2dvAtI.png',
                    'is_active' => 1,
                    'is_imei' => $data['is_imei'] ?? null,
                    'single_product_price' => $data['single_product_price'] ?? 0,
                    'group_order' => $data['group_order'] ?? null,
                    'alert_quantity' => $data['variants_alert_qty'][$key] ?? 0,
                ];
        
                $product = Product::create($variantData);
                $products[] = $product;
        
                // Save warehouse data
                Product_Warehouse::create([
                    "product_id" => $product->id,
                    "warehouse_id" => $data['warehouse'] ?? 1,
                    "qty" => $data["variants_qty"][$key] ?? 0,
                    "price" => $data["variants_sale_price"][$key] ?? 0
                ]);
            }
        
    

    } else {
        // Create a single product
        // dd($data);
        $product = Product::create($data);
        
        Product_Warehouse::create([
                        "product_id" => $product->id,
                        "warehouse_id" =>$data["warehouse"],
                        "qty" => $data["qty"],
                        "price" => $diff_price ?? 0 
                    ]);
       $products[] = $product;
                    
    }

    // Handle Barcode Generation
        $generator = new BarcodeGeneratorPNG();
        $barcodes = [];

        foreach ($products as $product) {
                    $barcodes[] = [
                        'color' => $product->color,
                        'size' => $product->size,
                        'code' => $product->code,
                         'qty'  => $product->qty,
                        'barcode' => "data:image/png;base64," . base64_encode($generator->getBarcode($product->id, $generator::TYPE_CODE_128)),
                    ];
            }
         return response()->json(['barcodes' => $barcodes]);
    }

    public function edit($id)
    {
        $role = Role::firstOrCreate(['id' => Auth::user()->role_id]);
        if ($role->hasPermissionTo('products-edit')) {
            $lims_product_list_without_variant = $this->productWithoutVariant();
            $lims_product_list_with_variant = $this->productWithVariant();
            $lims_brand_list = Brand::where('is_active', true)->get();
            $lims_category_list = Category::where('is_active', true)->get();
            $lims_unit_list = Unit::where('is_active', true)->get();
            $lims_tax_list = Tax::where('is_active', true)->get();
            
            // $lims_product_data = Product::where('id', $id)->first();

            $lims_product_data = Product::where('id', $id)->first();

            if ($lims_product_data) {
                $lims_product_data->name = Str::before($lims_product_data->name, '-');
                $lims_product_data->code = Str::before($lims_product_data->code, '-');
            }
            if ($lims_product_data->group_order) {
                $baseCode = explode('-', $lims_product_data->code)[0];
            
                $variantProducts = Product::with(['warehouses' => function($query) {
                        $query->select('warehouses.id'); // Customize columns if needed
                    }])
                    ->where('code', 'LIKE', $baseCode . '-%')
                    ->where('is_active', true)
                    ->get();
            }

            else {
                $variantProducts = collect();
            }

            $lims_product_variant_data = $lims_product_data->variant()->orderBy('position')->get();
            $lims_warehouse_list = Warehouse::where('is_active', true)->get();
            $colors = Color::where('is_active', true)->get();
            $sizes = Size::where('is_active',true)->get();
            return view('product.edit',compact('lims_product_list_without_variant', 'lims_product_list_with_variant', 'lims_brand_list', 'lims_category_list', 'lims_unit_list', 'lims_tax_list', 'lims_product_data', 'lims_product_variant_data', 'lims_warehouse_list','colors','sizes','variantProducts'));
        }
        else
            return redirect()->back()->with('not_permitted', 'Sorry! You are not allowed to access this module');
    }

    public function updateProduct(Request $request)
    {
       
        if(!env('USER_VERIFIED')) {
            \Session::flash('not_permitted', 'This feature is disable for demo!');
        }
        else {
            $this->validate($request, [
                'name' => [
                    'max:255',
                    // Rule::unique('products')->ignore($request->input('id'))->where(function ($query) {
                    //     return $query->where('is_active', 1);
                    // }),
                ],

                'code' => [
                    'max:255',
                    // Rule::unique('products')->ignore($request->input('id'))->where(function ($query) {
                    //     return $query->where('is_active', 1);
                    // }),
                ]
            ]);
            
            $lims_product_data = Product::findOrFail($request->input('id'));
            
          if ($request->has('group_order')) {
            $variantImages = $request->file('variants_image'); 
            $variantImagePaths = [];
        
            // Save uploaded images (if any)
            if (!empty($variantImages)) {
                foreach ($variantImages as $index => $variantImage) {
                    if ($variantImage) {
                        $imageName = time() . '_' . $variantImage->getClientOriginalName();
                        $variantImage->move(public_path('images/product'), $imageName);
                        $variantImagePaths[$index] = $imageName;
                    }
                }
            }
        
            // Make sure variants_id is an array before counting
            $variantIDs = $request->variants_id ?? [];
            $variantCount = count($variantIDs);
        
            // Assuming warehouse_id is an array and taking first warehouse by default
            $warehouseId = $request->warehouse_id[0] ?? null;
        
            for ($i = 0; $i < $variantCount; $i++) {
                $imageName = $variantImagePaths[$i] ?? null;
        
                $data = [
                    'type' => $request->type,
                    'name' => $request->variants_name[$i],
                    'color' => $request->variants_color[$i] ?? null,
                    'size' => $request->variants_size[$i] ?? null,
                    'code' => $request->variants_item_code[$i] ?? null,
                    'barcode_symbology' => $request->barcode_symbology,
                    'brand_id' => $request->brand_id ?? null,
                    'category_id' => $request->category_id ?? null,
                    'unit_id' => $request->unit_id,
                    'sale_unit_id' => $request->sale_unit_id,
                    'purchase_unit_id' => $request->purchase_unit_id,
                    'cost' => $request->variant_purchase_price[$i] ?? 0,
                    'price' => $request->variants_sale_price[$i] ?? 0,
                    'qty' => $request->variants_qty[$i] ?? 0,
                    'alert_quantity' => $request->variants_alert_qty[$i] ?? 0,
                    'tax_id' => $request->tax_id ?? null,
                    'tax_method' => $request->tax_method ?? null,
                    'featured' => $request->featured ?? 0,
                    'is_variant' => null,
                    'is_active' => 1,
                    'is_imei' => $request->is_imei ?? null,
                    'single_product_price' => $request->single_product_price ?? 0,
                    'group_order' => $request->group_order ?? null,
                ];
        
                if (!is_null($variantIDs[$i]) && $variantIDs[$i] !== "null") {
                    $product = Product::find($variantIDs[$i]);
                    if ($product) {
                        // Use new image if provided; else keep existing image or fallback default
                        $data['image'] = $imageName ?? $product->image ?? 'zummXD2dvAtI.png';
        
                        // Update existing product variant
                        $product->update($data);
        
                        // Update or create product warehouse entry
                        Product_Warehouse::updateOrCreate(
                            [
                                'product_id' => $product->id,
                                'warehouse_id' => $warehouseId,
                            ],
                            [
                                'qty' => $data['qty'],
                                'price' => $data['price'],
                            ]
                        );
                    } else {
                        // If product not found for some reason, maybe create new? (Optional)
                        $data['image'] = $imageName ?? 'zummXD2dvAtI.png';
                        $product = Product::create($data);
                        Product_Warehouse::create([
                            'product_id' => $product->id,
                            'warehouse_id' => $warehouseId,
                            'qty' => $data['qty'],
                            'price' => $data['price'],
                        ]);
                    }
                } else {
                    // New product variant creation
                    $data['image'] = $imageName ?? 'zummXD2dvAtI.png';
        
                    $product = Product::create($data);
        
                    Product_Warehouse::create([
                        'product_id' => $product->id,
                        'warehouse_id' => $warehouseId,
                        'qty' => $data['qty'],
                        'price' => $data['price'],
                    ]);
                }
            }
        }




            
            $data = $request->except('image', 'file', 'prev_img');
            $data['name'] = htmlspecialchars(trim($data['name']));

            if($data['type'] == 'combo') {
                $data['product_list'] = implode(",", $data['product_id']);
                $data['variant_list'] = implode(",", $data['variant_id']);
                $data['qty_list'] = implode(",", $data['product_qty']);
                $data['price_list'] = implode(",", $data['unit_price']);
                $data['cost'] = $data['unit_id'] = $data['purchase_unit_id'] = $data['sale_unit_id'] = 0;
            }
            elseif($data['type'] == 'digital' || $data['type'] == 'service')
                $data['cost'] = $data['unit_id'] = $data['purchase_unit_id'] = $data['sale_unit_id'] = 0;

            if(!isset($data['featured']))
                $data['featured'] = 0;

            if(!isset($data['promotion']))
                $data['promotion'] = null;

            if(!isset($data['is_batch']))
                $data['is_batch'] = null;

            if(!isset($data['is_imei']))
                $data['is_imei'] = null;

            $data['product_details'] = str_replace('"', '@', $data['product_details']);
            $data['product_details'] = $data['product_details'];
            if($data['starting_date'])
                $data['starting_date'] = date('Y-m-d', strtotime($data['starting_date']));
            if($data['last_date'])
                $data['last_date'] = date('Y-m-d', strtotime($data['last_date']));

            $previous_images = [];
            //dealing with previous images
            if($request->prev_img) {
                foreach ($request->prev_img as $key => $prev_img) {
                    if(!in_array($prev_img, $previous_images))
                        $previous_images[] = $prev_img;
                }
                $lims_product_data->image = implode(",", $previous_images);
                $lims_product_data->color =$request->color ?? null;
                $lims_product_data->save();
            }
            else {
                $lims_product_data->image = null;
                $lims_product_data->save();
            }

            //dealing with new images
            if($request->image) {
                $images = $request->image;
                $image_names = [];
                $length = count(explode(",", $lims_product_data->image));
                foreach ($images as $key => $image) {
                    $ext = pathinfo($image->getClientOriginalName(), PATHINFO_EXTENSION);
                    /*$image = Image::make($image)->resize(512, 512);*/
                    $imageName = date("Ymdhis") . ($length + $key+1) . '.' . $ext;
                    $image->move('public/images/product', $imageName);
                    $image_names[] = $imageName;
                }
                if($lims_product_data->image)
                    $data['image'] = $lims_product_data->image. ',' . implode(",", $image_names);
                else
                    $data['image'] = implode(",", $image_names);
            }
            else
                $data['image'] = $lims_product_data->image;

            $file = $request->file;
            if ($file) {
                $ext = pathinfo($file->getClientOriginalName(), PATHINFO_EXTENSION);
                $fileName = strtotime(date('Y-m-d H:i:s'));
                $fileName = $fileName . '.' . $ext;
                $file->move('public/product/files', $fileName);
                $data['file'] = $fileName;
            }

            // $lims_product_variant_data = ProductVariant::where('product_id', $request->input('id'))->select('id', 'variant_id')->get();
            // foreach ($lims_product_variant_data as $key => $value) {
            //     if (!in_array($value->variant_id, $data['variant_id'])) {
            //         ProductVariant::find($value->id)->delete();
            //     }
            // }
            if(isset($data['is_variant'])) {
                // foreach ($data['variant_name'] as $key => $variant_name) {
                //     if($data['product_variant_id'][$key] == 0) {
                //         $lims_variant_data = Variant::firstOrCreate(['name' => $data['variant_name'][$key]]);
                //         $lims_product_variant_data = new ProductVariant();

                //         $lims_product_variant_data->product_id = $lims_product_data->id;
                //         $lims_product_variant_data->variant_id = $lims_variant_data->id;

                //         $lims_product_variant_data->position = $key + 1;
                //         $lims_product_variant_data->item_code = $data['item_code'][$key];
                //         $lims_product_variant_data->additional_price = $data['additional_price'][$key];
                //         $lims_product_variant_data->qty = 0;
                //         $lims_product_variant_data->save();
                //     }
                //     else {
                //         Variant::find($data['variant_id'][$key])->update(['name' => $variant_name]);
                //         ProductVariant::find($data['product_variant_id'][$key])->update([
                //             'position' => $key+1,
                //             'item_code' => $data['item_code'][$key],
                //             'additional_price' => $data['additional_price'][$key]
                //         ]);
                //     }
                // }
            }
            else {
                $data['is_variant'] = null;
                $product_variants = ProductVariant::where('product_id', $lims_product_data->id)->get();
                foreach ($product_variants as $key => $product_variant) {
                    $product_variant->delete();
                }
            }
            if(isset($data['is_diffPrice'])) {
                foreach ($data['diff_price'] as $key => $diff_price) {
                    if($diff_price) {
                        $lims_product_warehouse_data = Product_Warehouse::FindProductWithoutVariant($lims_product_data->id, $data['warehouse_id'][$key])->first();
                        if($lims_product_warehouse_data) {
                            $lims_product_warehouse_data->price = $diff_price;
                            $lims_product_warehouse_data->save();
                        }
                        else {
                            Product_Warehouse::create([
                                "product_id" => $lims_product_data->id,
                                "warehouse_id" => $data["warehouse_id"][$key],
                                "qty" => 0,
                                "price" => $diff_price
                            ]);
                        }
                    }
                }
            }
            else {
                $data['is_diffPrice'] = false;
                foreach ($data['warehouse_id'] as $key => $warehouse_id) {
                    $lims_product_warehouse_data = Product_Warehouse::FindProductWithoutVariant($lims_product_data->id, $warehouse_id)->first();
                    if($lims_product_warehouse_data) {
                        $lims_product_warehouse_data->price = null;
                        $lims_product_warehouse_data->save();
                    }
                }
            }
            $lims_product_data->update($data);
            // \Session::flash('edit_message', 'Product updated successfully');
            return response()->json([
                'status' => 'success',
                'message' => 'Product updated successfully'
            ]);

        }
    }

    public function generateCode()
    {
        $id = Keygen::numeric(8)->generate();
        return $id;
    }

    public function search(Request $request)
    {
        $product_code = explode(" ", $request['data']);
        $lims_product_data = Product::where('code', $product_code[0])->first();

        $product[] = $lims_product_data->name;
        $product[] = $lims_product_data->code;
        $product[] = $lims_product_data->qty;
        $product[] = $lims_product_data->price;
        $product[] = $lims_product_data->id;
        return $product;
    }

    public function saleUnit($id)
    {
        $unit = Unit::where("base_unit", $id)->orWhere('id', $id)->pluck('unit_name','id');
        return json_encode($unit);
    }

    public function getData($id, $variant_id)
    {
        if($variant_id) {
            $data = Product::join('product_variants', 'products.id', 'product_variants.product_id')
                ->select('products.name', 'product_variants.item_code')
                ->where([
                    ['products.id', $id],
                    ['product_variants.variant_id', $variant_id]
                ])->first();
            $data->code = $data->item_code;
        }
        else
            $data = Product::select('name', 'code')->find($id);
        return $data;
    }

    // public function productWarehouseData($id)
    // {
    //     $warehouse = [];
    //     $qty = [];
    //     $batch = [];
    //     $expired_date = [];
    //     $imei_number = [];
    //     $warehouse_name = [];
    //     $variant_name = [];
    //     $variant_qty = [];
    //     $product_warehouse = [];
    //     $product_variant_warehouse = [];
    //     $lims_product_data = Product::select('id', 'is_variant')->find($id);
    //     if($lims_product_data->is_variant) {
    //         $lims_product_variant_warehouse_data = Product_Warehouse::where('product_id', $lims_product_data->id)->orderBy('warehouse_id')->get();
    //         $lims_product_warehouse_data = Product_Warehouse::select('warehouse_id', DB::raw('sum(qty) as qty'))->where('product_id', $id)->groupBy('warehouse_id')->get();
    //         foreach ($lims_product_variant_warehouse_data as $key => $product_variant_warehouse_data) {
    //             $lims_warehouse_data = Warehouse::find($product_variant_warehouse_data->warehouse_id);
    //             $lims_variant_data = Variant::find($product_variant_warehouse_data->variant_id);
    //             $warehouse_name[] = $lims_warehouse_data->name;
    //             $variant_name[] = $lims_variant_data->name;
    //             $variant_qty[] = $product_variant_warehouse_data->qty;
    //         }
    //     }
    //     else {
    //         $lims_product_warehouse_data = Product_Warehouse::where('product_id', $id)->orderBy('warehouse_id', 'asc')->get();
    //     }
    //     foreach ($lims_product_warehouse_data as $key => $product_warehouse_data) {
    //         $lims_warehouse_data = Warehouse::find($product_warehouse_data->warehouse_id);
    //         if($product_warehouse_data->product_batch_id) {
    //             $product_batch_data = ProductBatch::select('batch_no', 'expired_date')->find($product_warehouse_data->product_batch_id);
    //             $batch_no = $product_batch_data->batch_no;
    //             $expiredDate = date(config('date_format'), strtotime($product_batch_data->expired_date));
    //         }
    //         else {
    //             $batch_no = 'N/A';
    //             $expiredDate = 'N/A';
    //         }
    //         $warehouse[] = $lims_warehouse_data->name;
    //         $batch[] = $batch_no;
    //         $expired_date[] = $expiredDate;
    //         $qty[] = $product_warehouse_data->qty;
    //         if($product_warehouse_data->imei_number)
    //             $imei_number[] = $product_warehouse_data->imei_number;
    //         else
    //             $imei_number[] = 'N/A';
    //     }

    //     $product_warehouse = [$warehouse, $qty, $batch, $expired_date, $imei_number];
    //     $product_variant_warehouse = [$warehouse_name, $variant_name, $variant_qty];
    //     return ['product_warehouse' => $product_warehouse, 'product_variant_warehouse' => $product_variant_warehouse];
    // }

public function productWarehouseData($id)
{
    $warehouse = [];
    $qty = [];
    $batch = [];
    $expired_date = [];
    $imei_number = [];
    $warehouse_name = [];
    $variant_name = [];
    $variant_qty = [];
    $product_warehouse = [];
    $product_variant_warehouse = [];

    $baseProduct = Product::select('id', 'code', 'is_variant')->find($id);

    // Load base product warehouse data
    $lims_product_warehouse_data = Product_Warehouse::where('product_id', $baseProduct->id)
        ->orderBy('warehouse_id', 'asc')
        ->get();

    foreach ($lims_product_warehouse_data as $product_warehouse_data) {
        $lims_warehouse_data = Warehouse::find($product_warehouse_data->warehouse_id);

        if ($product_warehouse_data->product_batch_id) {
            $product_batch_data = ProductBatch::select('batch_no', 'expired_date')->find($product_warehouse_data->product_batch_id);
            $batch_no = $product_batch_data->batch_no ?? 'N/A';
            $expiredDate = $product_batch_data->expired_date 
                ? date(config('date_format'), strtotime($product_batch_data->expired_date)) 
                : 'N/A';
        } else {
            $batch_no = 'N/A';
            $expiredDate = 'N/A';
        }

        $warehouse[] = $lims_warehouse_data->name ?? 'N/A';
        $batch[] = $batch_no;
        $expired_date[] = $expiredDate;
        $qty[] = $product_warehouse_data->qty;
        $imei_number[] = $product_warehouse_data->imei_number ?? 'N/A';
    }

    // Load variant products from same table using code pattern
    if (Str::contains($baseProduct->code, '001')) {
        $baseCode = $baseProduct->code;
        $baseCode = explode('-', $baseCode)[0]; 
        // Fetch variant products whose code starts with base- (excluding the base itself)
        $variantProducts = Product::where('code', 'LIKE', $baseCode . '-%')
    ->where('is_active', true)
    ->get();

        foreach ($variantProducts as $variant) {
            $variantWarehouses = Product_Warehouse::where('product_id', $variant->id)
                ->orderBy('warehouse_id')
                ->get();

            foreach ($variantWarehouses as $vw) {
                $warehouseData = Warehouse::find($vw->warehouse_id);
                $warehouse_name[] = $warehouseData->name ?? 'N/A';

                // Variant name extracted from code (everything after base-)
                $variantLabel = trim(str_replace($baseCode . '-', '', $variant->code));
                $variant_name[] = $variantLabel ?: 'N/A';
                $variant_qty[] = $vw->qty;
            }
        }
    }

    $product_warehouse = [$warehouse, $qty, $batch, $expired_date, $imei_number];
    $product_variant_warehouse = [$warehouse_name, $variant_name, $variant_qty];

    return [
        'product_warehouse' => $product_warehouse,
        'product_variant_warehouse' => $product_variant_warehouse
    ];
}

    public function printBarcode()
    {
        $lims_product_list_without_variant = $this->productWithoutVariant();
        $lims_product_list_with_variant = $this->productWithVariant();
        $barcodesetting = BarcodeSetting::first();
        return view('product.print_barcode', compact('lims_product_list_without_variant', 'lims_product_list_with_variant','barcodesetting'));
    }

    public function productWithoutVariant()
    {
        return Product::ActiveStandard()->select('id', 'name', 'code')
                ->whereNull('is_variant')->get();
    }

    public function productWithVariant()
    {
        return Product::join('product_variants', 'products.id', 'product_variants.product_id')
                ->ActiveStandard()
                ->whereNotNull('is_variant')
                ->select('products.id', 'products.name', 'product_variants.item_code')
                ->orderBy('position')->get();
    }

    public function limsProductSearch(Request $request)
    {
        $product_code = explode("(", $request['data']);
        $product_code[0] = rtrim($product_code[0], " ");
        $lims_product_data = Product::where([
            ['code', $product_code[0] ],
            ['is_active', true]
        ])->first();
        if(!$lims_product_data) {
            $lims_product_data = Product::join('product_variants', 'products.id', 'product_variants.product_id')
                ->select('products.*', 'product_variants.item_code', 'product_variants.variant_id', 'product_variants.additional_price')
                ->where('product_variants.item_code', $product_code[0])
                ->first();

            $variant_id = $lims_product_data->variant_id;
            $additional_price = $lims_product_data->additional_price;
        }
        else {
            $variant_id = '';
            $additional_price = 0;
        }
        $product[] = $lims_product_data->name;
        if($lims_product_data->is_variant)
            $product[] = $lims_product_data->item_code;
        else
            $product[] = $lims_product_data->code;
        
        $product[] = $lims_product_data->price + $additional_price;
        $product[] = DNS1D::getBarcodePNG((string)$lims_product_data->id, $lims_product_data->barcode_symbology);
        $product[] = $lims_product_data->promotion_price;
        $product[] = config('currency');
        $product[] = config('currency_position');
        $product[] = $lims_product_data->qty;
        $product[] = $lims_product_data->id;
        $product[] = $variant_id;
        $product[] = $lims_product_data->color;
        $product[] = $lims_product_data->size;
        return $product;
    }

    /*public function getBarcode()
    {
        return DNS1D::getBarcodePNG('72782608', 'C128');
    }*/

    public function checkBatchAvailability($product_id, $batch_no, $warehouse_id)
    {
        $product_batch_data = ProductBatch::where([
            ['product_id', $product_id],
            ['batch_no', $batch_no]
        ])->first();
        if($product_batch_data) {
            $product_warehouse_data = Product_Warehouse::select('qty')
            ->where([
                ['product_batch_id', $product_batch_data->id],
                ['warehouse_id', $warehouse_id]
            ])->first();
            if($product_warehouse_data) {
                $data['qty'] = $product_warehouse_data->qty;
                $data['product_batch_id'] = $product_batch_data->id;
                $data['expired_date'] = date(config('date_format'), strtotime($product_batch_data->expired_date));
                $data['message'] = 'ok';
            }
            else {
                $data['qty'] = 0;
                $data['message'] = 'This Batch does not exist in the selected warehouse!';
            }            
        }
        else {
            $data['message'] = 'Wrong Batch Number!';
        }
        return $data;
    }

    public function importProduct(Request $request)
    {   
        //get file
        $upload=$request->file('file');
        $ext = pathinfo($upload->getClientOriginalName(), PATHINFO_EXTENSION);
        if($ext != 'csv')
            return redirect()->back()->with('message', 'Please upload a CSV file');

        $filePath=$upload->getRealPath();
        //open and read
        $file=fopen($filePath, 'r');
        $header= fgetcsv($file);
        $escapedHeader=[];
        //validate
        foreach ($header as $key => $value) {
            $lheader=strtolower($value);
            $escapedItem=preg_replace('/[^a-z]/', '', $lheader);
            array_push($escapedHeader, $escapedItem);
        }
        //looping through other columns
        while($columns=fgetcsv($file))
        {
            foreach ($columns as $key => $value) {
                $value=preg_replace('/\D/','',$value);
            }
           $data= array_combine($escapedHeader, $columns);
           
           if($data['brand'] != 'N/A' && $data['brand'] != ''){
                $lims_brand_data = Brand::firstOrCreate(['title' => $data['brand'], 'is_active' => true]);
                $brand_id = $lims_brand_data->id;
           }
            else
                $brand_id = null;

           $lims_category_data = Category::firstOrCreate(['name' => $data['category'], 'is_active' => true]);

           $lims_unit_data = Unit::where('unit_code', $data['unitcode'])->first();
           if(!$lims_unit_data)
                return redirect()->back()->with('not_permitted', 'Unit code does not exist in the database.');

           $product = Product::firstOrNew([ 'name'=>$data['name'], 'is_active'=>true ]);
            if($data['image'])
                $product->image = $data['image'];
            else
                $product->image = 'zummXD2dvAtI.png';

           $product->name = $data['name'];
           $product->code = $data['code'];
           $product->type = strtolower($data['type']);
           $product->barcode_symbology = 'C128';
           $product->brand_id = $brand_id;
           $product->category_id = $lims_category_data->id;
           $product->unit_id = $lims_unit_data->id;
           $product->purchase_unit_id = $lims_unit_data->id;
           $product->sale_unit_id = $lims_unit_data->id;
           $product->cost = $data['cost'];
           $product->price = $data['price'];
           $product->tax_method = 1;
           $product->qty = 0;
           $product->product_details = $data['productdetails'];
           $product->is_active = true;
           $product->save();

            if($data['variantname']) {
                //dealing with variants
                $variant_names = explode(",", $data['variantname']);
                $item_codes = explode(",", $data['itemcode']);
                $additional_prices = explode(",", $data['additionalprice']);
                foreach ($variant_names as $key => $variant_name) {
                    $variant = Variant::firstOrCreate(['name' => $variant_name]);
                    if($data['itemcode'])
                        $item_code = $item_codes[$key];
                    else
                        $item_code = $variant_name . '-' . $data['code'];
                    
                    if($data['additionalprice'])
                        $additional_price = $additional_prices[$key];
                    else
                        $additional_price = 0;

                    ProductVariant::create([
                        'product_id' => $product->id,
                        'variant_id' => $variant->id,
                        'position' => $key + 1,
                        'item_code' => $item_code,
                        'additional_price' => $additional_price,
                        'qty' => 0
                    ]);
                }
                $product->is_variant = true;
                $product->save();
            }
         }
         return redirect('products')->with('import_message', 'Product imported successfully');
    }

    public function deleteBySelection(Request $request)
    {
        $product_id = $request['productIdArray'];
        foreach ($product_id as $id) {
            $lims_product_data = Product::findOrFail($id);
            $lims_product_data->is_active = false;
            $lims_product_data->save();
        }
        return 'Product deleted successfully!';
    }

    // public function destroy($id)
    // {
    //     $lims_product_data = Product::findOrFail($id);
    //     $lims_product_data->is_active = false;
    //     if($lims_product_data->image != 'zummXD2dvAtI.png') {
    //         $images = explode(",", $lims_product_data->image);
    //         foreach ($images as $key => $image) {
    //             if(file_exists('public/images/product/'.$image))
    //                 unlink('public/images/product/'.$image);
    //         }
    //     }
    //     $lims_product_data->save();
    //     return redirect('products')->with('message', 'Product deleted successfully');
    // }
    public function destroy(Request $request, $id)
    {
        try {
            $product = Product::findOrFail($id);
            $product->is_active = false;
    
            // Delete images if not default
            if ($product->image !== 'zummXD2dvAtI.png') {
                $images = explode(",", $product->image);
                foreach ($images as $image) {
                    $path = public_path('images/product/' . $image);
                    if (file_exists($path)) {
                        unlink($path);
                    }
                }
            }
    
            $product->save();
    
            // If API request (expects JSON)
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Product deleted successfully',
                    'product_id' => $product->id
                ]);
            }
    
            // If normal form submission (HTML)
            return redirect('products')->with('message', 'Product deleted successfully');
    
        } catch (\Exception $e) {
            // Error response for API
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error deleting product',
                    'error' => $e->getMessage()
                ], 500);
            }
    
            // Error response for web form
            return redirect()->back()->withErrors('Error deleting product: ' . $e->getMessage());
        }
    }
    public function deleteVariant($id)
    {
        try {
            $product = Product::findOrFail($id);
            $product->is_active = false;
    
            // Delete product images (if not default image)
            if ($product->image !== 'zummXD2dvAtI.png') {
                $images = explode(",", $product->image);
                foreach ($images as $image) {
                    $path = public_path('images/product/' . $image);
                    if (file_exists($path)) {
                        unlink($path);
                    }
                }
            }
    
            $product->save();
    
            return response()->json([
                'success' => true,
                'message' => 'Product deleted successfully',
                'product_id' => $product->id
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting product',
                'error' => $e->getMessage()
            ], 500);
        }
    }


}
