<?php

namespace App\Http\Controllers;

use App\WhatsappEvoProfile;
use Illuminate\Http\Request;
use App\Services\EvoService;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;


class WhatsappEvolutionProfileController extends Controller
{
      public function index(Request $request)
    {
        $id = auth()->user()->id;
        $search = $request->input('search');
        $isActive = $request->input('is_active');
    
        $query = WhatsappEvoProfile::where('user_id', auth()->id());
        
        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'LIKE', "%{$search}%")
                  ->orWhere('Instance', 'LIKE', "%{$search}%"); 
            });
        }
    
        if (!is_null($isActive) && $isActive !== '') {
            $query->where('is_active', $isActive);
        }
    
        $profiles = $query->orderBy('created_at', 'desc')->paginate(10);
    
        return view('evo-whatsapp-profiles.index', compact('profiles'));
    }


    public function create()
    {
        return view('evo-whatsapp-profiles.create');
    }

    public function store(Request $request, EvoService $whatsappService)
    {
           $validator = Validator::make($request->all(), [
                    'name'  => 'required|string|max:255',
                    'phone' => [
                    'required',
                    'string',
                    'max:20',
                    Rule::unique('whatsapp_evo_profiles', 'phone')->where(fn($q) => $q->where('is_active', 1)),
                ],
            ], [
                'phone.unique' => 'This number is already in use',
            ]);
    
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
                'message' => 'Validation failed',
            ], 422);
        }
    
        
        try {
            $apiResponse = $whatsappService->createInstance($request->phone);

            $profile = WhatsappEvoProfile::create([
                'user_id'   => auth()->id(),
                'name'      => $request->name,
                'phone'     => $request->phone,
                'instance'   => $apiResponse['instance']['instanceName'] ?? null,
                'qrcode'    => $apiResponse['qrcode']['base64'] ?? null,
                'pairing_code'  => $apiResponse['qrcode']['pairingCode'] ?? null,
                'is_active' => 0,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Profile created successfully',
                'data'    => $profile,
                'api'     => $apiResponse,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    
     public function checkConnection(string $instance, EvoService $whatsappService)
    {
        try {
            $apiResponse = $whatsappService->checkConnection($instance);

            $isActive = ($apiResponse['instance']['state'] === 'open') ? 1 : 0;

            WhatsappEvoProfile::where('instance', $instance)
                ->update(['is_active' => $isActive]);

            return response()->json([
                'success'   => true,
                'connected' => $isActive === 1,
                'message'   => $isActive 
                                ? 'Profile connected successfully' 
                                : 'Profile connection failed',
                'api'       => $apiResponse,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success'   => false,
                'connected' => false,
                'message'   => $e->getMessage(),
            ], 500);
        }
    }



    public function show($encryptedId)
    {
         $id = Crypt::decrypt($encryptedId); 
        $profile = WhatsappEvoProfile::findOrFail($id);
        return view('evo-whatsapp-profiles.show', compact('profile'));
    }
    public function edit($encryptedId)
    {
        $id = Crypt::decrypt($encryptedId); 
        $profile = WhatsappEvoProfile::findOrFail($id);

        return view('evo-whatsapp-profiles.update', compact('profile'));
    }

    public function update(Request $request, string $id)
    {
        $profile = WhatsappEvoProfile::findOrFail($id);

        $request->validate([
            'name'  => 'required|string|max:255',
            'phone' => 'required|string|max:20|unique:whatsapp_evo_profiles,phone,' . $profile->id,
        ]);

        $profile->update([
            'name'  => $request->name,
            'phone' => $request->phone,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Profile updated successfully',
            'data'    => $profile
        ]);
    }

    public function destroy($encryptedId, EvoService $whatsappService)
    {

        $id = Crypt::decrypt($encryptedId);
        $profile = WhatsappEvoProfile::findOrFail($id);
        $apiResponse = $whatsappService->deleteInstance($profile->instance);
        $profile->delete();

        return back()->with([
                'success' =>'Profile deleted successfully'
            ]);
    }
}
