<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class EvoService
{
    /**
     * Get Evo API settings from .env
     */
    protected static function getSettings(): array
    {
        return [
            'baseUrl' => env('WHATSAPP_EVO_BASE_URL', ''), // Your Evo API base URL
            'apiKey'  => env('WHATSAPP_EVO_API_KEY', ''),  // Your Evo API key
        ];
    }

    /**
     * Send a text message
     */
    public static function sendText(string $instanceId, string $phone, string $message)
    {
        $settings = self::getSettings();

        $payload = [
            'number' => preg_replace('/\D/', '', $phone),
            'text'   => $message,
        ];

        $endpoint = "{$settings['baseUrl']}/message/sendText/{$instanceId}";

        return self::sendRequest($endpoint, $payload, $phone, $settings['apiKey']);
    }

    /**
     * Send media (document, image, etc.)
     */
    public static function sendMedia(string $instanceId, string $phone, string $message, string $mediaUrl, string $mediaType = 'image')
    {
        $settings = self::getSettings();

         $payload = [
            'number'    => preg_replace('/\D/', '', $phone),
            'mediatype' => $mediaType,
            'caption'   => $message,
            'media'     => $mediaUrl,
            'fileName'  => basename(parse_url($mediaUrl, PHP_URL_PATH)),
        ];

        $endpoint = "{$settings['baseUrl']}/message/sendMedia/{$instanceId}";

        return self::sendRequest($endpoint, $payload, $phone, $settings['apiKey']);
    }

    /**
     * Internal request handler
     */
    protected static function sendRequest(string $endpoint, array $payload, string $phone, string $apiKey)
    {
        try {
            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
                'apikey'       => $apiKey,
            ])->post($endpoint, $payload);

            return $response->json();
        } catch (\Exception $e) {
            Log::error("WhatsApp API Error for {$phone}: " . $e->getMessage());
            return [
                'status'   => 'failed',
                'response' => ['error' => $e->getMessage()],
            ];
        }
    }

    /**
     * Create a new WhatsApp instance
     */
    public static function createInstance(string $phone): array
    {
        $settings = self::getSettings();
        $instanceName = 'EVO-' . uniqid();

        $response = Http::withHeaders([
            'apikey' => $settings['apiKey'],
            'Accept' => 'application/json',
        ])->post("{$settings['baseUrl']}/instance/create", [
            'instanceName' => $instanceName,
            'number'       => $phone,
            'qrcode'       => true,
            'integration'  => 'WHATSAPP-BAILEYS',
        ]);

        if ($response->failed()) {
            throw new \Exception('Failed to create WhatsApp instance: ' . $response->body());
        }

        return $response->json();
    }

    /**
     * Check connection status of an instance
     */
    public static function checkConnection(string $instance): array
    {
        $settings = self::getSettings();

        $response = Http::withHeaders([
            'apikey' => $settings['apiKey'],
            'Accept' => 'application/json',
        ])->get("{$settings['baseUrl']}/instance/connectionState/{$instance}");

        return $response->json();
    }

    /**
     * Delete an instance
     */
    public static function deleteInstance(string $instance): array
    {
        $settings = self::getSettings();

        $response = Http::withHeaders([
            'apikey' => $settings['apiKey'],
            'Accept' => 'application/json',
        ])->get("{$settings['baseUrl']}/instance/delete/{$instance}");

        return $response->json();
    }
}
